/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { isEmpty } from "lodash-es";
import { BehaviorSubject, combineLatest, combineLatestWith, of, } from "rxjs";
import { catchError, distinctUntilChanged, finalize, map, take, tap, } from "rxjs/operators";
import { inject, Injectable } from "@angular/core";
import { deepClone, isVoid, } from "common";
import { NavigationStore } from "../../../../store/navigation/navigation.store";
import { RecordSaveGQL } from "../../../../store/record/graphql/api.record.save";
import { LanguageStore } from "../../../../store/language/language.store";
import { ModuleNavigation } from "../../../../services/navigation/module-navigation/module-navigation.service";
import { MetadataStore, } from "../../../../store/metadata/metadata.store.service";
import { MessageService } from "../../../../services/message/message.service";
import { AppStateStore } from "../../../../store/app-state/app-state.store";
import { RecordManager } from "../../../../services/record/record.manager";
import { LocalStorageService } from "../../../../services/local-storage/local-storage.service";
import { SubpanelStoreFactory } from "../../../../containers/subpanel/store/subpanel/subpanel.store.factory";
import { ViewStore } from "../../../../store/view/view.store";
import { RecordFetchGQL } from "../../../../store/record/graphql/api.record.get";
import { StatisticsBatch } from "../../../../store/statistics/statistics-batch.service";
import { RecordStoreFactory } from "../../../../store/record/record.store.factory";
import { UserPreferenceStore } from "../../../../store/user-preference/user-preference.store";
import { PanelLogicManager } from "../../../../components/panel-logic/panel-logic.manager";
import { RecordConvertService } from "../../../../services/record/record-convert.service";
import { FieldActionsAdapterFactory } from "../../../../components/field-layout/adapters/field.actions.adapter.factory";
import { RecordValidationHandler } from "../../../../services/record/validation/record-validation.handler";
import * as i0 from "@angular/core";
import * as i1 from "../../../../store/record/graphql/api.record.get";
import * as i2 from "../../../../store/record/graphql/api.record.save";
import * as i3 from "../../../../store/app-state/app-state.store";
import * as i4 from "../../../../store/language/language.store";
import * as i5 from "../../../../store/navigation/navigation.store";
import * as i6 from "../../../../services/navigation/module-navigation/module-navigation.service";
import * as i7 from "../../../../store/metadata/metadata.store.service";
import * as i8 from "../../../../services/local-storage/local-storage.service";
import * as i9 from "../../../../services/message/message.service";
import * as i10 from "../../../../containers/subpanel/store/subpanel/subpanel.store.factory";
import * as i11 from "../../../../services/record/record.manager";
import * as i12 from "../../../../store/statistics/statistics-batch.service";
import * as i13 from "../../../../store/record/record.store.factory";
import * as i14 from "../../../../store/user-preference/user-preference.store";
import * as i15 from "../../../../components/panel-logic/panel-logic.manager";
import * as i16 from "../../../../services/record/record-convert.service";
const initialState = {
    module: "",
    recordID: "",
    loading: false,
    widgets: false,
    showSidebarWidgets: false,
    showTopWidget: false,
    showSubpanels: false,
    mode: "detail",
    params: {
        returnModule: "",
        returnId: "",
        returnAction: "",
    },
};
class RecordViewStore extends ViewStore {
    constructor(recordFetchGQL, recordSaveGQL, appStateStore, languageStore, navigationStore, moduleNavigation, metadataStore, localStorage, message, subpanelFactory, recordManager, statisticsBatch, recordStoreFactory, preferences, panelLogicManager, recordConvertService) {
        super(appStateStore, languageStore, navigationStore, moduleNavigation, metadataStore);
        this.recordFetchGQL = recordFetchGQL;
        this.recordSaveGQL = recordSaveGQL;
        this.appStateStore = appStateStore;
        this.languageStore = languageStore;
        this.navigationStore = navigationStore;
        this.moduleNavigation = moduleNavigation;
        this.metadataStore = metadataStore;
        this.localStorage = localStorage;
        this.message = message;
        this.subpanelFactory = subpanelFactory;
        this.recordManager = recordManager;
        this.statisticsBatch = statisticsBatch;
        this.recordStoreFactory = recordStoreFactory;
        this.preferences = preferences;
        this.panelLogicManager = panelLogicManager;
        this.recordConvertService = recordConvertService;
        this.panels = [];
        /** Internal Properties */
        this.cache$ = null;
        this.internalState = deepClone(initialState);
        this.store = new BehaviorSubject(this.internalState);
        this.state$ = this.store.asObservable();
        this.subpanelReloadSubject = new BehaviorSubject({});
        this.subpanelReloadSub = [];
        this.subs = [];
        this.fieldSubs = [];
        this.panelsSubject = new BehaviorSubject(this.panels);
        this.viewModeSubject = new BehaviorSubject("detail");
        this.viewMode$ = this.viewModeSubject.asObservable();
        this.actionAdaptorFactory = inject(FieldActionsAdapterFactory);
        this.panels$ = this.panelsSubject.asObservable();
        this.recordStore = recordStoreFactory.create(this.getViewFieldsObservable(), this.getRecordMetadata$());
        this.record$ = this.recordStore.state$.pipe(distinctUntilChanged());
        this.stagingRecord$ = this.recordStore.staging$.pipe(distinctUntilChanged());
        this.loading$ = this.state$.pipe(map((state) => state.loading));
        this.widgets$ = this.state$.pipe(map((state) => state.widgets));
        this.showSidebarWidgets$ = this.state$.pipe(map((state) => state.showSidebarWidgets));
        this.showTopWidget$ = this.state$.pipe(map((state) => state.showTopWidget));
        this.showSubpanels$ = this.state$.pipe(map((state) => state.showSubpanels));
        this.mode$ = this.state$.pipe(map((state) => state.mode));
        this.subpanelReload$ = this.subpanelReloadSubject.asObservable();
        const data$ = this.record$.pipe(combineLatestWith(this.loading$), map(([record, loading]) => {
            this.data = { record, loading };
            return this.data;
        }));
        this.vm$ = data$.pipe(combineLatestWith(this.appData$, this.metadata$), map(([data, appData, metadata]) => {
            this.vm = { data, appData, metadata };
            return this.vm;
        }));
        this.subpanelsState = new BehaviorSubject({});
        this.subpanels$ = this.subpanelsState.asObservable();
        this.viewContext$ = this.record$.pipe(map(() => this.getViewContext()));
        this.initPanels();
        this.recordValidationHandler = inject(RecordValidationHandler);
    }
    get widgets() {
        return this.internalState.widgets;
    }
    set widgets(show) {
        this.updateState({
            ...this.internalState,
            widgets: show,
        });
    }
    get showSidebarWidgets() {
        return this.internalState.showSidebarWidgets;
    }
    set showSidebarWidgets(show) {
        this.savePreference(this.getModuleName(), "show-sidebar-widgets", show);
        this.updateState({
            ...this.internalState,
            showSidebarWidgets: show,
        });
    }
    get showTopWidget() {
        return this.internalState.showTopWidget;
    }
    set showTopWidget(show) {
        this.updateState({
            ...this.internalState,
            showTopWidget: show,
        });
    }
    get showSubpanels() {
        return this.internalState.showTopWidget;
    }
    set showSubpanels(show) {
        this.updateState({
            ...this.internalState,
            showSubpanels: show,
        });
    }
    get params() {
        return this.internalState.params || {};
    }
    set params(params) {
        this.updateState({
            ...this.internalState,
            params,
        });
    }
    getModuleName() {
        return this.internalState.module;
    }
    getRecordId() {
        return this.internalState.recordID;
    }
    getViewContext() {
        return {
            module: this.getModuleName(),
            id: this.getRecordId(),
            record: this.getBaseRecord(),
        };
    }
    getSubpanels() {
        return this.subpanels;
    }
    /**
     * Clean destroy
     */
    destroy() {
        this.clear();
    }
    /**
     * Initial record load if not cached and update state.
     * Returns observable to be used in resolver if needed
     *
     * @param {string} module to use
     * @param {string} recordID to use
     * @param {string} mode to use
     * @param {object} params to set
     * @returns {object} Observable<any>
     */
    init(module, recordID, mode = "detail", params = {}) {
        this.internalState.module = module;
        this.internalState.recordID = recordID;
        this.setMode(mode);
        this.initSubpanels(module, recordID);
        this.calculateShowWidgets();
        return this.load().pipe(tap(() => {
            this.showTopWidget = true;
            this.loadSubpanelStatistics(module);
            this.parseParams(params);
        }));
    }
    /**
     * Clear observable cache
     */
    clear() {
        this.cache$ = null;
        this.clearSubpanels();
        this.subpanelsState.unsubscribe();
        this.updateState(deepClone(initialState));
        this.subs = this.safeUnsubscription(this.subs);
        this.fieldSubs = this.safeUnsubscription(this.fieldSubs);
    }
    /**
     * Get staging record
     *
     * @returns {string} ViewMode
     */
    getBaseRecord() {
        if (!this.internalState) {
            return null;
        }
        return this.recordStore.getBaseRecord();
    }
    /**
     * Get current view mode
     *
     * @returns {string} ViewMode
     */
    getMode() {
        if (!this.internalState) {
            return null;
        }
        return this.internalState.mode;
    }
    /**
     * Set new mode
     *
     * @param {string} mode ViewMode
     */
    setMode(mode) {
        this.updateState({ ...this.internalState, mode });
    }
    save() {
        this.appStateStore.updateLoading(`${this.internalState.module}-record-save`, true);
        this.updateState({
            ...this.internalState,
            loading: true,
        });
        return this.recordStore.save().pipe(catchError(() => {
            this.message.addDangerMessageByKey("LBL_ERROR_SAVING");
            return of({});
        }), finalize(() => {
            this.setMode("detail");
            this.appStateStore.updateLoading(`${this.internalState.module}-record-save`, false);
            this.updateState({
                ...this.internalState,
                loading: false,
            });
        }));
    }
    saveOnEdit() {
        return this.recordStore.save().pipe(catchError(() => {
            this.message.addDangerMessageByKey("LBL_ERROR_SAVING");
            return of({});
        }), finalize(() => {
            this.appStateStore.updateLoading(`${this.internalState.module}-record-save`, false);
            this.updateState({
                ...this.internalState,
                loading: false,
            });
        }));
    }
    /**
     * Load / reload record using current pagination and criteria
     *
     * @param {boolean} useCache if to use cache
     * @returns {object} Observable<RecordViewState>
     */
    load(useCache = true) {
        this.updateState({
            ...this.internalState,
            loading: true,
        });
        return this.recordStore
            .retrieveRecord(this.internalState.module, this.internalState.recordID, useCache)
            .pipe(tap((data) => {
            this.updateState({
                ...this.internalState,
                recordID: data.id,
                module: data.module,
                loading: false,
            });
        }));
    }
    /**
     * Get summary template
     *
     * @returns {string} summary template label
     */
    getSummaryTemplate() {
        const metadata = this.metadata || {};
        const recordMeta = metadata.recordView || {};
        const templates = recordMeta.summaryTemplates || {};
        return templates[this.getMode()] || "";
    }
    /**
     * Parse query params
     *
     * @param {object} params to set
     */
    parseParams(params = {}) {
        if (!params) {
            return;
        }
        const currentParams = { ...this.internalState.params };
        Object.keys(params).forEach((paramKey) => {
            if (!isVoid(currentParams[paramKey])) {
                currentParams[paramKey] = params[paramKey];
                return;
            }
        });
        this.params = params;
    }
    /**
     * Load all statistics
     *
     * @param {string} module if to use cache
     */
    loadSubpanelStatistics(module) {
        const subpanels = this.subpanelsState.value;
        if (!subpanels) {
            return;
        }
        const queries = {};
        Object.keys(subpanels).forEach((subpanelKey) => {
            const subpanel = subpanels[subpanelKey];
            const statsMap = subpanel.statistics;
            if (!statsMap || !Object.keys(statsMap).length) {
                return;
            }
            if (subpanel.shouldBatchStatistic() === false) {
                subpanel.loadAllStatistics().pipe(take(1)).subscribe();
                return;
            }
            const subpanelQueries = subpanel.getAllStatisticQuery();
            Object.keys(subpanelQueries).forEach((subpanelQueryKey) => {
                const queryKey = this.buildStatKey(subpanelKey, subpanelQueryKey);
                queries[queryKey] = subpanelQueries[subpanelQueryKey];
            });
            subpanel.setAllStatisticsLoading(true);
        });
        this.statisticsBatch
            .fetch(module, queries)
            .pipe(take(1))
            .subscribe((stats) => {
            Object.keys(subpanels).forEach((subpanelKey) => {
                const subpanel = subpanels[subpanelKey];
                const subpanelQueries = subpanel.getAllStatisticQuery();
                Object.keys(subpanelQueries).forEach((subpanelQueryKey) => {
                    const queryKey = this.buildStatKey(subpanelKey, subpanelQueryKey);
                    const stat = stats[queryKey];
                    if (!stat) {
                        return;
                    }
                    subpanel.setStatistics(subpanelQueryKey, stat, true);
                });
                subpanel.setAllStatisticsLoading(false);
            });
        });
    }
    buildStatKey(subpanelKey, subpanelQueryKey) {
        subpanelKey = subpanelKey.replace(/_/g, "-");
        subpanelQueryKey = subpanelQueryKey.replace(/_/g, "-");
        return subpanelKey + "-" + subpanelQueryKey;
    }
    /**
     * Update the state
     *
     * @param {object} state to set
     */
    updateState(state) {
        const previousMode = this.internalState?.mode;
        this.internalState = state;
        this.store.next(state);
        if (state.mode && state.mode !== previousMode) {
            console.log("[record-view.store.ts] Mode changed:", state.mode);
            this.viewModeSubject.next(state.mode); // ✅ emit new mode
        }
    }
    /**
     * Init subpanels
     *
     * @param {string} module parent module
     * @param {string} recordId id
     */
    initSubpanels(module, recordId) {
        this.showSubpanels = true;
        this.metadataStore.subPanelMetadata$.subscribe((meta) => {
            this.clearSubpanels();
            Object.keys(meta).forEach((key) => {
                this.subpanels[key] = this.subpanelFactory.create();
                this.subpanels[key].init(module, recordId, meta[key], this.record$);
            });
            this.subpanelsState.next(this.subpanels);
            Object.keys(this.subpanels).forEach((subpanelKey) => {
                const subpanel = this.subpanels[subpanelKey];
                this.subpanelReloadSub.push(subpanel.recordList.records$
                    .pipe(tap(() => {
                    const update = {};
                    update[subpanelKey] = true;
                    this.subpanelReloadSubject.next(update);
                }))
                    .subscribe());
            });
        });
    }
    initPanels() {
        const panelSub = combineLatest([
            this.metadataStore.recordViewMetadata$,
            this.stagingRecord$,
            this.languageStore.vm$,
        ]).subscribe(([meta, record, languages]) => {
            const panels = [];
            const module = (record && record.module) || "";
            this.safeUnsubscription(this.fieldSubs);
            meta.panels.forEach((panelDefinition) => {
                const label = panelDefinition.label
                    ? panelDefinition.label.toUpperCase()
                    : this.languageStore.getFieldLabel(panelDefinition.key.toUpperCase(), module, languages);
                const panel = { label, key: panelDefinition.key, rows: [] };
                let adaptor = null;
                const tabDef = meta.templateMeta.tabDefs[panelDefinition.key.toUpperCase()] ?? null;
                if (tabDef) {
                    panel.meta = tabDef;
                }
                panelDefinition.rows.forEach((rowDefinition) => {
                    const row = { cols: [] };
                    rowDefinition.cols.forEach((cellDefinition) => {
                        const cellDef = { ...cellDefinition };
                        const fieldActions = cellDefinition.fieldActions || null;
                        if (fieldActions) {
                            adaptor = this.actionAdaptorFactory.create("recordView", cellDef.name, this);
                            cellDef.adaptor = adaptor;
                        }
                        row.cols.push(cellDef);
                    });
                    panel.rows.push(row);
                });
                panel.displayState = new BehaviorSubject(tabDef?.display ?? true);
                panel.display$ = panel.displayState.asObservable();
                panels.push(panel);
                if (isEmpty(record?.fields) || isEmpty(tabDef?.displayLogic)) {
                    return;
                }
                Object.values(tabDef.displayLogic).forEach((logicDef) => {
                    if (isEmpty(logicDef?.params?.fieldDependencies)) {
                        return;
                    }
                    logicDef.params.fieldDependencies.forEach((fieldKey) => {
                        const field = record.fields[fieldKey] || null;
                        if (isEmpty(field)) {
                            return;
                        }
                        this.fieldSubs.push(field.valueChanges$.subscribe(() => {
                            this.panelLogicManager.runLogic(logicDef.key, field, panel, record, this.getMode());
                        }));
                    });
                });
            });
            this.panelsSubject.next((this.panels = panels));
            return panels;
        });
        this.subs.push(panelSub);
    }
    clearSubpanels() {
        if (this.subpanels) {
            Object.keys(this.subpanels).forEach((key) => {
                this.subpanels[key].clear();
            });
        }
        if (this.subpanelReloadSub.length) {
            this.subpanelReloadSub.forEach((sub) => sub.unsubscribe());
            this.subpanelReloadSub = [];
        }
        this.subpanels = {};
    }
    /**
     * Calculate if widgets are to display
     */
    calculateShowWidgets() {
        let show = false;
        const recordViewMeta = this.getRecordViewMetadata();
        const sidebarWidgetsConfig = recordViewMeta.sidebarWidgets || [];
        if (sidebarWidgetsConfig && sidebarWidgetsConfig.length > 0) {
            show = true;
        }
        const showSidebarWidgets = this.loadPreference(this.getModuleName(), "show-sidebar-widgets") ?? null;
        if (showSidebarWidgets !== null) {
            this.showSidebarWidgets = showSidebarWidgets;
        }
        else {
            this.showSidebarWidgets = show;
        }
        this.widgets = show;
    }
    /**
     * Get record view metadata
     *
     * @returns {object} metadata RecordViewMetadata
     */
    getRecordViewMetadata() {
        const meta = this.metadataStore.get() || {};
        return meta.recordView || {};
    }
    /**
     * Get vardefs
     *
     * @returns {object} vardefs FieldDefinitionMap
     */
    getVardefs() {
        const meta = this.getRecordViewMetadata();
        return meta.vardefs || {};
    }
    /**
     * Get view fields observable
     *
     * @returns {object} Observable<ViewFieldDefinition[]>
     */
    getViewFieldsObservable() {
        return this.metadataStore.recordViewMetadata$.pipe(map((recordMetadata) => {
            const fieldsMap = {};
            recordMetadata.panels.forEach((panel) => {
                panel.rows.forEach((row) => {
                    row.cols.forEach((col) => {
                        const fieldName = col.name ?? col.fieldDefinition.name ?? "";
                        fieldsMap[fieldName] = col;
                    });
                });
            });
            Object.keys(recordMetadata.vardefs).forEach((fieldKey) => {
                const vardef = recordMetadata.vardefs[fieldKey] ?? null;
                if (!vardef || isEmpty(vardef)) {
                    return;
                }
                // already defined. skip
                if (fieldsMap[fieldKey]) {
                    return;
                }
                if (vardef.type == "relate") {
                    return;
                }
                fieldsMap[fieldKey] = {
                    name: fieldKey,
                    vardefBased: true,
                    label: vardef.vname ?? "",
                    type: vardef.type ?? "",
                    display: vardef.display ?? "",
                    fieldDefinition: vardef,
                    metadata: vardef.metadata ?? {},
                    logic: vardef.logic ?? {},
                };
            });
            return Object.values(fieldsMap);
        }));
    }
    getRecordMetadata$() {
        return this.metadataStore.recordViewMetadata$.pipe(map((recordMetadata) => {
            return recordMetadata?.metadata ?? {};
        }));
    }
    /**
     * Build ui user preference key
     *
     * @param {string} storageKey Storage Key
     * @protected
     * @returns {string} Preference Key
     */
    getPreferenceKey(storageKey) {
        return "recordview-" + storageKey;
    }
    /**
     * Save ui user preference
     *
     * @param {string} module Module
     * @param {string} storageKey Storage Key
     * @param {any} value Value
     * @protected
     */
    savePreference(module, storageKey, value) {
        this.preferences.setUi(module, this.getPreferenceKey(storageKey), value);
    }
    /**
     * Load ui user preference
     *
     * @param {string} module Module
     * @param {string} storageKey Storage Key
     * @protected
     * @returns {any} User Preference
     */
    loadPreference(module, storageKey) {
        return this.preferences.getUi(module, this.getPreferenceKey(storageKey));
    }
    safeUnsubscription(subscriptionArray) {
        subscriptionArray.forEach((sub) => {
            if (sub.closed) {
                return;
            }
            sub.unsubscribe();
        });
        subscriptionArray = [];
        return subscriptionArray;
    }
    setViewMode(mode) {
        this.viewModeSubject.next(mode);
    }
    static { this.ɵfac = function RecordViewStore_Factory(t) { return new (t || RecordViewStore)(i0.ɵɵinject(i1.RecordFetchGQL), i0.ɵɵinject(i2.RecordSaveGQL), i0.ɵɵinject(i3.AppStateStore), i0.ɵɵinject(i4.LanguageStore), i0.ɵɵinject(i5.NavigationStore), i0.ɵɵinject(i6.ModuleNavigation), i0.ɵɵinject(i7.MetadataStore), i0.ɵɵinject(i8.LocalStorageService), i0.ɵɵinject(i9.MessageService), i0.ɵɵinject(i10.SubpanelStoreFactory), i0.ɵɵinject(i11.RecordManager), i0.ɵɵinject(i12.StatisticsBatch), i0.ɵɵinject(i13.RecordStoreFactory), i0.ɵɵinject(i14.UserPreferenceStore), i0.ɵɵinject(i15.PanelLogicManager), i0.ɵɵinject(i16.RecordConvertService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: RecordViewStore, factory: RecordViewStore.ɵfac }); }
}
export { RecordViewStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(RecordViewStore, [{
        type: Injectable
    }], function () { return [{ type: i1.RecordFetchGQL }, { type: i2.RecordSaveGQL }, { type: i3.AppStateStore }, { type: i4.LanguageStore }, { type: i5.NavigationStore }, { type: i6.ModuleNavigation }, { type: i7.MetadataStore }, { type: i8.LocalStorageService }, { type: i9.MessageService }, { type: i10.SubpanelStoreFactory }, { type: i11.RecordManager }, { type: i12.StatisticsBatch }, { type: i13.RecordStoreFactory }, { type: i14.UserPreferenceStore }, { type: i15.PanelLogicManager }, { type: i16.RecordConvertService }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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