/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Injectable } from '@angular/core';
import { Router, UrlTree } from '@angular/router';
import { forkJoin, of } from 'rxjs';
import { catchError, filter, map, take, tap } from 'rxjs/operators';
import { AuthService } from './auth.service';
import { AsyncActionService } from '../process/processes/async-action/async-action';
import { AppStateStore } from '../../store/app-state/app-state.store';
import { RouteConverter } from '../navigation/route-converter/route-converter.service';
import { emptyObject, isEmptyString } from 'common';
import { LanguageStore } from '../../store/language/language.store';
import { NotificationStore } from '../../store/notification/notification.store';
import * as i0 from "@angular/core";
import * as i1 from "@angular/router";
import * as i2 from "./auth.service";
import * as i3 from "../process/processes/async-action/async-action";
import * as i4 from "../../store/app-state/app-state.store";
import * as i5 from "../navigation/route-converter/route-converter.service";
import * as i6 from "../../store/language/language.store";
import * as i7 from "../../store/notification/notification.store";
class AuthGuard {
    constructor(router, authService, asyncActionService, appState, routeConverter, language, notificationStore) {
        this.router = router;
        this.authService = authService;
        this.asyncActionService = asyncActionService;
        this.appState = appState;
        this.routeConverter = routeConverter;
        this.language = language;
        this.notificationStore = notificationStore;
    }
    canActivate(route, snapshot) {
        return this.authorizeUser(route, snapshot);
    }
    /**
     * Authorize user session and acl together in conjunction
     *
     * @returns {object} Observable<boolean | UrlTree> | Promise<boolean | UrlTree> | boolean | UrlTree
     * @param {ActivatedRouteSnapshot} route information about the current route
     * @param snapshot
     */
    authorizeUser(route, snapshot) {
        // Note: this session and acl are not always booleans
        return forkJoin([
            this.authorizeUserSession(route, snapshot),
            this.authorizeUserACL(route)
        ]).pipe(map(([session, acl]) => {
            if (session instanceof UrlTree) {
                return session;
            }
            if (acl instanceof UrlTree) {
                return acl;
            }
            return session && acl;
        }));
    }
    /**
     * Authorize user acl
     *
     * @returns {object} Observable<boolean | UrlTree>
     * @param {ActivatedRouteSnapshot} activatedRoute information about the current route
     */
    authorizeUserACL(activatedRoute) {
        const routeInfo = this.routeConverter.parseRouteURL(activatedRoute.url);
        const routeURL = this.appState.getRouteUrl() ?? '';
        if (!routeInfo.module || routeInfo.module === 'home') {
            return of(true);
        }
        const homeUrl = '';
        const homeUrlTree = this.router.parseUrl(homeUrl);
        const actionName = 'user-acl';
        const asyncData = {
            action: actionName,
            module: routeInfo.module,
            payload: {
                routeAction: routeInfo.action,
                record: routeInfo.record,
                routeURL,
                queryParams: activatedRoute?.queryParams ?? []
            }
        };
        return this.asyncActionService.run(actionName, asyncData)
            .pipe(take(1), map((process) => {
            if (process.data && process.data.result === true) {
                return true;
            }
            if (isEmptyString(routeURL)) {
                // Re-direct to home
                return homeUrlTree;
            }
            const currentUrlTree = this.router.parseUrl(this.router.url);
            if (this.routeConverter.isClassicViewRoute(currentUrlTree)) {
                return currentUrlTree;
            }
            return false;
        }), catchError(() => of(homeUrlTree)));
    }
    /**
     * Authorize user session
     *
     * @returns {object} Observable<boolean | UrlTree> | Promise<boolean | UrlTree> | boolean | UrlTree
     * @param {ActivatedRouteSnapshot} route information about the current route
     * @param snapshot
     */
    authorizeUserSession(route, snapshot) {
        if (this.authService.isUserLoggedIn.value && route.data.checkSession !== true) {
            return of(true);
        }
        let sessionExpiredUrl = this.authService.getSessionExpiredRoute();
        const redirect = this.authService.sessionExpiredRedirect();
        const sessionExpiredUrlTree = this.router.parseUrl(sessionExpiredUrl);
        return this.authService.fetchSessionStatus()
            .pipe(take(1), map((user) => {
            if (user && user.appStatus.installed === false) {
                return this.router.parseUrl('install');
            }
            if (user && user.active === true) {
                const wasLoggedIn = !!this.appState.getCurrentUser();
                this.authService.setCurrentUser(user);
                if (!wasLoggedIn) {
                    this.language.appStrings$.pipe(filter(appStrings => appStrings && !emptyObject(appStrings)), tap(() => {
                        this.notificationStore.enableNotifications();
                        this.notificationStore.refreshNotifications();
                    }), take(1)).subscribe();
                }
                if (user?.redirect?.route && (!snapshot.url.includes(user.redirect.route))) {
                    const redirectUrlTree = this.router.parseUrl(user.redirect.route);
                    redirectUrlTree.queryParams = user?.redirect?.queryParams ?? {};
                    return redirectUrlTree;
                }
                return true;
            }
            this.appState.setPreLoginUrl(snapshot.url);
            this.authService.resetState();
            if (redirect) {
                this.authService.handleSessionExpiredRedirect();
                return false;
            }
            // Re-direct to login
            return sessionExpiredUrlTree;
        }), catchError(() => {
            if (redirect) {
                this.authService.handleSessionExpiredRedirect();
                return of(false);
            }
            this.authService.logout('LBL_SESSION_EXPIRED', false);
            return of(sessionExpiredUrlTree);
        }), tap((result) => {
            if (result === true) {
                this.authService.isUserLoggedIn.next(true);
            }
        }));
    }
    static { this.ɵfac = function AuthGuard_Factory(t) { return new (t || AuthGuard)(i0.ɵɵinject(i1.Router), i0.ɵɵinject(i2.AuthService), i0.ɵɵinject(i3.AsyncActionService), i0.ɵɵinject(i4.AppStateStore), i0.ɵɵinject(i5.RouteConverter), i0.ɵɵinject(i6.LanguageStore), i0.ɵɵinject(i7.NotificationStore)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: AuthGuard, factory: AuthGuard.ɵfac, providedIn: 'root' }); }
}
export { AuthGuard };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(AuthGuard, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], function () { return [{ type: i1.Router }, { type: i2.AuthService }, { type: i3.AsyncActionService }, { type: i4.AppStateStore }, { type: i5.RouteConverter }, { type: i6.LanguageStore }, { type: i7.NotificationStore }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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